
#include "SPI_ArduinoMaster.h"

void SPI_ArduinoMaster::begin(uint8_t chipSelectPin)
{
	TimeoutMs = SPIMASTER_TIMEOUTMS_DEFAULT;

	// Запоминаем пин чип селекта
	_chipSelectPin = chipSelectPin;

	// Настраиваем пин чип селекта
	pinMode(_chipSelectPin, OUTPUT);
	digitalWrite(_chipSelectPin, HIGH);

	// Устанавливаем скорость в 250кбит/сек (16МГц / 64)
	SPI.setClockDivider(SPI_CLOCK_DIV64);

	// Устанавливаем полярность и фазу тактовых импульсов
	SPI.setDataMode(SPI_MODE0);

	// Отправляем старшим байтом вперед
	SPI.setBitOrder(MSBFIRST);

	// Активируем SPI
	SPI.begin();
}

bool SPI_ArduinoMaster::SendBytes(uint8_t* arr, uint8_t arrSize)
{
	// Проверяем корректность параметров
	if (arrSize > 254)
		return false;

	DesyncFlag = 0;

	_SetChipSelect();

	// 1. Отправляем SYN
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte(SPIMASTER_SYN))
		return false;

	// 2. Отправляем байт направления данных
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte(SPIMASTER_SEND))
	{
		DesyncFlag = 1;
		return false;
	}

	// 3. Отправляем длину отправляемого пакета
	// Отправляем заданный байт до тех пор пока не получим ненулевой ответ
	if (!_SendTheByteAndWaitNotNullResponse(arrSize, &_rx))
		return _FailTransaction();
	// Прверяем совпадает ли принятый слейвом байт с оригинальным
	if (!_EqualReceivedByte(_rx, arrSize))
		return false;
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte())
		return false;

	// Чисто декоративны момент, на работу не влияет
	_delay_us(50);

	//4. Отправляем информационные байты
	for (i = 0; i < arrSize; i++)
		_ByteExchange(arr[i]);
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte())
		return false;

	// 5. Отправляем CRC пакета
	_CRC = _CalcCRC(arr, arrSize);
	// Отправляем заданный байт до тех пор пока не получим ненулевой ответ
	if (!_SendTheByteAndWaitNotNullResponse(_CRC, &_rx))
		return _FailTransaction();
	// Прверяем совпадает ли принятый слейвом байт с оригинальным
	if (!_EqualReceivedByte(_rx, _CRC))
		return false;
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte())
		return false;

	// 6. Ждем байта корректности принятых данных
	// Отправляем заданный байт до тех пор пока не получим ненулевой ответ
	if (!_SendTheByteAndWaitNotNullResponse(0, &_rx))
		return _FailTransaction();
	if (_rx == SPIMASTER_ACK)
	{
		// Ждем подтверждения получения
		if (!_ACK_AfterReceivingByte())
			return false;
	}
	else
		return _FailTransaction();

	
	_ResetChipSelect();
	return true;
}
bool SPI_ArduinoMaster::ReceiveBytes(uint8_t* arr, uint8_t* arrSize)
{
	DesyncFlag = 0;

	_SetChipSelect();

	// 1. Отправляем SYN
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte(SPIMASTER_SYN))
		return false;

	// 2. Отправляем байт направления данных
	// Ждем подтверждения получения
	if (!_ACK_AfterSendingByte(SPIMASTER_RECEIVE))
	{
		DesyncFlag = 1;
		return true;
	}
	
	// 3. Получаем длину принимаемого пакета
	if (!_SendTheByteAndWaitNotNullResponse(0, arrSize))
		return _FailTransaction();
	// Отправляем ее обратно для проверки корректности получения
	if (!_SendByteToEqual(*arrSize))
		return false;
	// Ждем подтверждения получения
	if (!_ACK_AfterReceivingByte())
		return false;
	
	// 4. Получаем информационные байты
	for(i = 0; i < *arrSize; i++)
		arr[i] = _ByteExchange(0);
	// Подтверждаем получение
	if (!_ACK_AfterReceivingByte())
		return false;

	// 5. Получаем CRC
	if (!_SendTheByteAndWaitNotNullResponse(0, &_CRC))
		return _FailTransaction();
	// Отправляем ее обратно для проверки корректности получения
	if (!_SendByteToEqual(_CRC))
		return false;
	// Ждем подтверждения получения
	if (!_ACK_AfterReceivingByte())
		return false;
	
	// 6. Проверяем полученные байты по CRC
	if (_CRC == _CalcCRC(arr, *arrSize))
	{
		_ByteExchange(SPIMASTER_ACK);
		// Ждем подтверждения получения
		if (!_ACK_AfterSendingByte())
			return false;
	}
	else
		return _FailTransaction();


	_ResetChipSelect();
	return true;
}

bool SPI_ArduinoMaster::_ACK_AfterSendingByte(void)
{
	// Ждем подтверждения получения
	if (!_SendTheByteAndWaitTheResponse(0, SPIMASTER_ACK))
		return _FailTransaction();

	// Отправляем ACK в ответ
	_ByteExchange(SPIMASTER_ACK);

	// Ждем подтверждения получения
	if (!_SendTheByteAndWaitTheResponse(0, SPIMASTER_ACK))
		return _FailTransaction();

	return true;
}
bool SPI_ArduinoMaster::_ACK_AfterSendingByte(uint8_t byteToSend)
{
	// Ждем подтверждения получения
	if (!_SendTheByteAndWaitTheResponse(byteToSend, SPIMASTER_ACK))
		return _FailTransaction();

	// Отправляем ACK в ответ
	_ByteExchange(SPIMASTER_ACK);

	// Ждем подтверждения получения
	if (!_SendTheByteAndWaitTheResponse(0, SPIMASTER_ACK))
		return _FailTransaction();

	return true;
}
bool SPI_ArduinoMaster::_ACK_AfterReceivingByte(void)
{
	// Отправляем ACK в ответ
	_ByteExchange(SPIMASTER_ACK);

	// Ждем подтверждения получения
	if (!_SendTheByteAndWaitTheResponse(0, SPIMASTER_ACK))
		return _FailTransaction();

	// Отправляем ACK в ответ
	_ByteExchange(SPIMASTER_ACK);

	return true;
}

bool SPI_ArduinoMaster::_SendByteToEqual(uint8_t tx)
{
	_ByteExchange(tx);

	// Ждем байта корректного получения байта
	if (!_SendTheByteAndWaitTheResponse(0, SPIMASTER_ACK))
		return _FailTransaction();

	return true;
}
bool SPI_ArduinoMaster::_EqualReceivedByte(uint8_t rx, uint8_t equalWith)
{
	// Проверяем совпадает ли принятый байт rx с байтом equalWith
	if (rx == equalWith)
	{
		// Отправляем байт правильности принятого байта
		_ByteExchange(SPIMASTER_ACK);

		return true;
	}
	else return _FailTransaction();
}

void SPI_ArduinoMaster::_SetChipSelect(void)
{
	digitalWrite(_chipSelectPin, LOW);
}
void SPI_ArduinoMaster::_ResetChipSelect(void)
{
	digitalWrite(_chipSelectPin, HIGH);
}

bool SPI_ArduinoMaster::_FailTransaction()
{
	_ResetChipSelect();
	_SetChipSelect();
	_ByteExchange(SPIMASTER_NACK); //_SendTheByteAndWaitTheResponse(SPIMASTER_NACK, SPIMASTER_ACK, SPIMASTER_TIMEOUTMS_DEFAULT / 2);
	_ResetChipSelect();
	return false;
}

uint8_t SPI_ArduinoMaster::_ByteExchange(uint8_t tx, unsigned long timeoutMs)
{
	//Serial.println(">> _ByteExchange");
	_ByteExchange_StartTime = millis();
	SPDR = tx;
	do
	{
		if (millis() - _ByteExchange_StartTime > timeoutMs)
			return 0;
		//Serial.println("Wait for ending transaction ...");
	} while (!(SPSR & (1 << SPIF)));
	_ByteExchange_Bufer = SPDR;
	//Serial.println("<< _ByteExchange");
	return _ByteExchange_Bufer;
}

bool SPI_ArduinoMaster::_SendTheByteAndWaitTheResponse(uint8_t tx, uint8_t rx, unsigned long timeoutMs)
{
	_rx = 0;

	if (timeoutMs > 0)
	{
		_timeoutCheckStartTime = millis();
		do
		{
			if (millis() - _timeoutCheckStartTime > timeoutMs)
				return false;
			_rx = _ByteExchange(tx);
			if(_rx == SPIMASTER_NACK)
				return false;
		}while (_rx != rx);
		return true;
	}
	else
	{
		do
		{
			_rx = _ByteExchange(tx);
			if (_rx == SPIMASTER_NACK)
				return false;
		}while (_rx != rx);
		return true;
	}
}
bool SPI_ArduinoMaster::_SendTheByteAndWaitNotNullResponse(uint8_t tx, uint8_t* rx, unsigned long timeoutMs)
{
	_rx = 0;

	if (timeoutMs > 0)
	{
		_timeoutCheckStartTime = millis();
		do
		{
			if (millis() - _timeoutCheckStartTime > timeoutMs)
				return false;
			_rx = _ByteExchange(tx);
			if (_rx == SPIMASTER_NACK)
				return false;
		} while (_rx == 0);
		*rx = _rx;
		return true;
	}
	else
	{
		do
		{
			_rx = _ByteExchange(tx);
			if (_rx == SPIMASTER_NACK)
				return false;
		} while (_rx == 0);
		*rx = _rx;
		return true;
	}
}

uint8_t SPI_ArduinoMaster::_CalcCRC(uint8_t* data, size_t len)
{
	uint8_t crc = SPIMASTER_CRC8_SEED;
	size_t i, j;
	for (i = 0; i < len; i++) {
		crc ^= data[i];
		for (j = 0; j < 8; j++) {
			if ((crc & 0x80) != 0)
				crc = (uint8_t)((crc << 1) ^ SPIMASTER_CRC8_POLY);
			else
				crc <<= 1;
		}
	}

	i = 0;
	while (crc == SPIMASTER_SYN ||
		crc == SPIMASTER_ACK ||
		crc == SPIMASTER_NACK ||
		crc == SPIMASTER_SEND ||
		crc == SPIMASTER_RECEIVE ||
		crc == 0)
	{
		crc ^= data[i++];
		for (j = 0; j < 8; j++) {
			if ((crc & 0x80) != 0)
				crc = (uint8_t)((crc << 1) ^ SPIMASTER_CRC8_POLY);
			else
				crc <<= 1;
		}
	}

	return crc;
}
void SPI_ArduinoMaster::_delay_us(unsigned long time)
{
	unsigned long StartTime = micros();
	while (micros() - StartTime < time);
}
